unit ControlConnection;
{
    UNIT ControlConnection;
    Version number 2.61(out of beta)

This unit contains the control connection manager TControlConnection class.
All methods are described in the interface part.

Notes:
    * TControlConnection must re-instanted on every control connection.
    * Version 2.5 contains improved reply handling - see OnUnexpectedReply and
      onControlClosed. Added SetExpectReply().
    * Version 2.51: Added SendIPAndSynchSignals() and "TimeOutTimer:= nil;"
      line added to method ShutDownIO(), it was buggy - GPF rules :)
    * Version 2.52: Bug fixed with unexpected replies.
    * Version 2.6: Added Quick Terminate support to avoid AV's when the
      instanted class shut down (immediately exit from wait cycles).
    * Version 2.61: Added Winsock error saving in ShutDownIO(). Socket closing
      was override the original error codes.

Created by Pter Karsai.
}

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, WSocket, Forms, ExtCtrls,
     WinSock;

const ReadBufferSize = 1514;
{ ReadBufferSize, by the nature of ReceiveServerReply() limits the
  maximal line length of a server reply line. It does not mean, set
  it higher than 255 will make the program free to handle longer lines
  than 255 characters, but it means, if you set it below 255, all incoming
  lines will be fragmented and the routine will handle them incorrectly.
  Now this receive buffer is set to 1514, to avoid Ethernet packet
  fragmentation. }

{ Server reply constants }
const srPOS_PRELIMINARY_REPLY  = 1; { positive preliminary reply }
      srPOS_COMPLETION_REPLY   = 2; { positive completion reply }
      srPOS_INTERMEDIATE_REPLY = 3; { positive intermediate reply }
      srNEG_TRANSIENT_COMP_RPY = 4; { transient negative completion reply }
      srNEG_PERMANENT_COMP_RPY = 5; { permanent negative completion reply }
      srUNKNOWN_REPLY_TYPE     = 255; { check it out }

type
{ Ver 2.0:
  * The most of socket-level exceptions removed. I had to do this because of
    the number of the various exceptions can occur when handling sockets.
    To substitute their function, ECSocketException defined. It's Message
    property contains the error code provided by TWSocket.
  * All exceptions prefixed with EC... why? why not! }

  ECSocketException    = class(Exception);
  { thrown if any socket exception occured (onError and onBgException events) }
  ECNotConnected       = class(Exception);
  { thrown if you try to send/receive data from control connection while it's
    not connected. }
  ECConnectingTimedOut = class(Exception);
  { thrown if connecting to server timed out (on user level defined timeout!)}
  ECReplyTimedOut      = class(Exception);
  { thrown if no reply data arrives in the specified user-level time limit }

  TOutDial = procedure of object;  { type TOutDial simply call-type for
                                     external parameter-given procedures }

  TControlConnection = class(TObject)
  private
  { internal data }
    ControlSocket  : TWSocket;    { socket of the control connection }
    ServerReply    : TStringList; { server reply storage }
    RemoteHost     : string;      { host to connect to }
    RemotePort     : string;      { port on remote host }
    Connected      : boolean;     { represents current connection state }
    TimeOutTimer   : TTimer;      { internal timer to check timeouts }
    LastReplyCode  : word;        { last reply code by server }
    ReadBufferTop  : integer;     { top of the read buffer in ReceiveS...}
    Terminating    : boolean;     { termination signal for quick shutdown }

  { sychronizer variables }
    ExpectedReply        : boolean;  { TRUE if an operation need reply }
    ReceivingReply       : boolean;  { TRUE receiving is in progress }
    ReplyGot             : boolean;  { TRUE if a complete reply got }

  { internal communication }
    wsLastError  : word;          { last TWSocket error }
    TimeSpent    : word;          { counter variable }

{ ----------------------------------------------------------------------------}
{ timeout timer management }
{ ----------------------------------------------------------------------------}
    procedure StartTimer(TimeOut: byte);
{ StartTimer starts the timeout timer. }

    procedure StopTimer;
{ StopTimer stops the timeout timer. }

    function HasTimedOut(TimeOut: byte): boolean;
{ Returns TRUE, if current operation timed out, otherwise FALSE }

    procedure ShutDownIO;
{ ShutDownIO() method close socket I/O device and free it }

  public
{ ----------------------------------------------------------------------------}
{ Variable OnUnexpectedReply will called each time if an unexpected server
  message arrive. What's an unexpected reply? Simplier to say, what is not.
  Not unexpected the first reply after a command sent (see SendCommand()) and
  not after connecting, in any other case it's unexpected. You sure get such
  message, after data transfers. (commands RETR, STOR, STOU, APPE generate a
  'transfer started' reply first and when finish the transfer, another one about
  this fact). }
    OnUnexpectedReply : TOutDial;
    OnControlClosed   : TOutDial;
{ OnControlClosed will called if the control connection closed. }

{ ----------------------------------------------------------------------------}
{ control session management }
{ ----------------------------------------------------------------------------}
    procedure ConnectToServer(TimeOut: byte);
{ Function: Connects to the remote server given in the parameters of
  the TControlConnection class' constructor.

  * You can control the maximal time to wait for the connection. If the TimeOut
    parameter is 0, then no timeout check defined, else the ConnectToServer will
    wait max. TimeOut seconds before return back.

  * Note: After connecting, the FTP servers usually send some welcoming message.
    Use GetServerReply() to get this message. If you don't do this, you'll loose
    this message - sometimes it contains important informations, e.g. access
    info, rate data, u/l, d/l restrictions and so on.

  Throws ECSocketException on any connect error
    * short note: TWSocket doesn't throw any exceptions if no service presented
                  on remote host (WSACONNREFUSED) - I don't know why.
  Throws ECConnectingTimedOut on connecting time steps over the TimeOut par.
}

    function AmIConnected: boolean;
{ Function: Returns TRUE if the session is connected, otherwise FALSE }

    procedure CloseConnection;
{ Function: Closes the control connection immediately. Not same as QUIT cmd. }

    procedure SendCommand(command: string);
{ Function: Sends a string 'command' thru the socket...
  It's an asynchronous operation and does not wait for the server reply, if you
  want to wait, just call WaitForServerReply.
  Throws ECNotConnected if control socket not connected. }

    procedure SendIPAndSynchSignals;
{ Function: Sends an IP and Synch TELNET signal thru the control connection.
  It's required for commands to interpret them by the server while data
  transfer is in progress. }

{ ----------------------------------------------------------------------------}
{ event handlers }
{ ----------------------------------------------------------------------------}
    procedure ControlSessionConnected(Sender: TObject; Error: Word);
    procedure ControlDataAvailable(Sender: TObject; Error: Word);
    procedure ControlSessionClosed(Sender: TObject; Error: Word);
    procedure ControlSessionError(Sender: TObject);
    procedure TimeOutTimerTick(Sender: TObject);
    { TimeOutTimerTick used by determine whether an operation timed out }

{ ----------------------------------------------------------------------------}
{ server reply manager methods }
{ ----------------------------------------------------------------------------}
    procedure ReceiveServerReply;
{ Function: Receives the server reply. It is called by ControlDataAvailable()
  event to handle any incoming data from the server.
  It saves the incoming lines to ServerReply:TStringList string list. }

    function GetServerReply(TimeOut: byte): TStringList;
{ Function: Returns the last server reply. It's a synchronized method, waits
  until current server message completely arrive, then return.
  * A REPLY CAN BE READ *ONCE*. Re-call of GetServerReply() will cause
    TControlConnection to wait another server message. Ok, it's a
    programmer-was-a-big-ass solution, but I had to do it so because of the
    synchronization.
  * With TimeOut parameter, you can set up a limit as in ConnectToServer().
    Meaning of TimeOut here the limit of seconds between *each* incoming data
    packet.
  * Note: GetServerReply()'s result should be saved immendiately, because the
    next server reply will overwrite it!

  Throws ECSocketException on any connect socket error.
  Throws ECReplyTimedOut as the aforesaid conditions.
  Throws ECNotConnected if socket not connected.}

    function GetServerReplyCode: word;
{ Function: Returns the last server reply code }

    function GetServerReplyType: byte;
{ Function: Returns the last server code type as defined in srX* constants }

    procedure SetExpectReply(ExpectReply: boolean);
{ Function: By set ExpectReply, you can control whether call OnUnexpectedReply.
  By setting it to TRUE, no OnUnExpectedReply will called on the next server
  reply.
  Notes:* Don't care to set it before ConnectToServer() and SendCommand(), it
          will done automatically by those methods.
        * Default state for accepting replies is *unexpected*. So if you
          successfully received a reply, ExpectReply will automatically switched
          back to state 'unexpected'.
}

{ ----------------------------------------------------------------------------}
{ constructor and destructor }
{ ----------------------------------------------------------------------------}
    constructor Create(xRemoteHost, xRemotePort: string; AOwner: TComponent);
    destructor Destroy; override;
    procedure Free;
end;

implementation

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ timeout timer management ----------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
procedure TControlConnection.StartTimer(TimeOut: byte);
begin
{ prepare timeOutTimer timer }
     timeSpent:= 0;
     if TimeOut > 0 then
        TimeOutTimer.Interval:= 1000 { interval given in milliseconds }
     else
        TimeOutTimer.Interval:= 0; { on interval 0, no timer called }

{ start timer }
     TimeOutTimer.Enabled:= true;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.StopTimer;
begin
     TimeOutTimer.Enabled:= false;
end;

{------------------------------------------------------------------------------}

function TControlConnection.HasTimedOut(TimeOut: byte): boolean;
begin
     if Assigned(TimeOutTimer) then begin
     { return TRUE if timer started... }
        Result:= TimeOutTimer.Enabled and (TimeOutTimer.Interval > 0);
     { if timer started and time spent not reach limit, return FALSE }
        if Result then
           if TimeSpent <= TimeOut then Result:= false
     end
     else
        Result:= true;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.ShutDownIO;
var lastWSAError: integer;  { last Winsock error }
begin
{ save last Winsock error, because the socket closing will override it }
     LastWSAError:= WSAGetLastError;

{ sign methods that termination is in progress }
     Terminating:= true;
{ close and free ControlSocket if it's possible }
     if Assigned(ControlSocket) then begin
        if ControlSocket.State = wsConnected then ControlSocket.Close;
        ControlSocket.Free;
        ControlSocket:= nil;
     end;

{ stop timer and free if required }
     if Assigned(TimeOutTimer) then begin
        if TimeOutTimer.Enabled then TimeOutTimer.Enabled:= false;
        TimeOutTimer.Free;
        TimeOutTimer:= nil;
     end;

{ we don't wait for a reply anymore }
     ExpectedReply:= false;

{ set last Winsock error back }
     WSASetLastError(lastWSAError);
end;


{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ control session management --------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
procedure TControlConnection.ConnectToServer(TimeOut: byte);
begin
{ yeah... we're waiting for a reply }
     ExpectedReply:= true;
{ start timer }
     StartTimer(TimeOut);

{ Try to connect, Connect() will throw an exception if there are errors while
  connecting, for example 'No Route To Host' }
     try
        ControlSocket.Connect;
     except
        on Exception do begin
        { on any exception, just throw an ECSocketException away }
         ShutDownIO;
         raise ECSocketException.CreateFmt('(%d) %s',
               [WSAGetLastError, WSocketErrorDesc(WSAGetLastError)]);
        end;
     end;

{ Since Connect() is asynchronous, the following routine will wait while
  the session is connecting or the application terminated
  Routine taken from Francois Piette's CLIDEMO. Thanx! }
     while not Connected and not HasTimedOut(TimeOut) and (wsLastError = 0)
           and ((WSAGetLastError = 0) or (WSAGetLastError = WSAEWOULDBLOCK))
           and not Terminating do
     begin
         Application.ProcessMessages;
         if Application.Terminated then begin ShutDownIO; Exit end;
     end;
     if Terminating then exit;

{ stop timer }
     StopTimer;
{ if we've reached the TimeOut limit, throw exception ECConnectingTimedOut }
     if TimeSpent >= TimeOut then begin
        ShutDownIO;
        raise ECConnectingTimedOut.Create('Connecting to server timed out.');
     end;

{ if there was an error while connecting, for example 'Connection refused',
  'cause the requested service not implemented on remote host, throw
   ECSocketException with the error code replied by TWSocket or WinSock. }
      if (WSAGetLastError > 0) and (WSAGetLastError <> WSAEWOULDBLOCK) then
      begin
         ShutDownIO;
         raise ECSocketException.CreateFmt('Control socket error %d: %s',
               [WSAGetLastError, WSocketErrorDesc(WSAGetLastError)]);
      end;

      if not Connected then begin
         ShutDownIO;
         raise ECSocketException.Create(IntToStr(wsLastError));
      end;

end;

{------------------------------------------------------------------------------}

function TControlConnection.AmIConnected: boolean;
begin
     Result:= Connected;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.CloseConnection;
begin
{ disconnect from the server by the hard way }
     ControlSocket.Close;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.SendCommand(command: string);
begin
{ if socket is connected, send command + CRLF }
     if Connected then begin
        ExpectedReply:= true;
        ControlSocket.SendStr(command + #13#10)
     end
{ if not connected, throw exception ECNotConnected }
     else
        raise ECNotConnected.Create('Can''t send command - control connection'+
              ' is closed.');
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.SendIPAndSynchSignals;
begin
{ send an IP + DM(1) signal first - 244 and 242 ASCII }
     if Connected then begin
        ExpectedReply:= true;
        ControlSocket.SendStr(#244#242);
{ Synch signal: DM signal in a normal TCP packet and another DM in TCP_URGENT }
        ControlSocket.SendFlags:= wsSendUrgent;
        ControlSocket.SendStr(#242);  { DM(2) }
        ControlSocket.SendFlags:= wsSendNormal
     end
     else
        raise ECNotConnected.Create('Can''t send signals - control connection'+
              ' is closed.');

end;
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ event handler methods -------------------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

procedure TControlConnection.ControlSessionConnected(Sender: TObject;
                                                     Error: Word);
begin
{ errors suxx!, prepare to throw exception ECannotConnect (in ConnectToServer)}
     if Error > 0 then begin
        Connected:= false;
        wsLastError:= Error end
     else
        Connected:= true;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.ControlSessionClosed(Sender: TObject; Error: Word);
begin
{ if the session is closed, set the 'connected' state to FALSE immediately }
     wsLastError:= Error;
     Connected:= false;
{ call external procedure if defined }
     if Assigned(OnControlClosed) then OnControlClosed;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.ControlDataAvailable(Sender: TObject; Error: Word);
begin
{ if there's no error, save incoming data by ReceiveServerReply... }
     if Error = 0 then
     begin
        ReceiveServerReply      { not just say... do it :) }
     end
{ ... and on any error, prepare to throw ECSocketException in GetServerReply()
 I haven't experienced errors during the read yet, so maybe you'll be the first
 one who can see, this routine is buggy :) }
     else
     begin
        wsLastError:= Error;
        ReceivingReply:= false; { stop receiving on any error }
     end;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.ControlSessionError(Sender: TObject);
begin
     wsLastError:= ControlSocket.LastError;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.TimeOutTimerTick(Sender: TObject);
begin
     if TimeSpent < $ffff then inc(TimeSpent);
end;


{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ server reply manager methods  -----------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

procedure TControlConnection.ReceiveServerReply;
var readBuffer          : array[0..ReadBufferSize - 1] of char;
    receivedBytes, cPos : integer;
    processString       : string;
    saveChar            : char;
    LineGot             : boolean;
begin
{ receive the avaiable data to buffer 'readBuffer', the number of received
  bytes will appear in 'receivedBytes' }
    fillchar(readBuffer[ReadBufferTop], sizeOf(readBuffer) - ReadBufferTop - 1,
             #0);
    receivedBytes:= ControlSocket.Receive(@readBuffer[ReadBufferTop],
                                        sizeOf(readBuffer) - 1 - ReadBufferTop);

{ I've experienced a strange thing - when send QUIT command, so the socket will
  be closed soon - the FTP server send three or more packets, some of them
  doesn't contains data; I don't know why. If no data arrives, we don't want
  to process the string... }
    if receivedBytes <= 0 then Exit;

{ if we've another reply in the buffer, don't let it delete... anyway clear. }
    if not ReceivingReply then
       ServerReply.Clear
    else
       ExpectedReply:= true;

    ReplyGot:= false;  { we don't have complete reply yet }
    ReceivingReply:= true;  { sure, we're receiving a reply }
    LineGot:= false; { no reply - no line }

{ restart TimeOut value, because new data arrived }
    TimeSpent:= 0;

{ where's Wally? :) }
    ReadBufferTop:= ReadBufferTop + receivedBytes;

{ process incoming data }
    while true do begin
        cPos:= 0;
    { save all available strings !}
        while (readBuffer[cPos] <> #10) and (cPos < ReadBufferTop) do
              inc(cPos);
    { if no more lines in the buffer or line not received successfully }
        if cPos = ReadBufferTop then
        begin
             if not LineGot then ReadBufferTop:= cPos
             else ReadBufferTop:= 0;
             Exit;
        end;

    { tricky boy; #0 terminated string, doesn't look like a PChar? :)}
        saveChar:= readBuffer[cPos];
        readBuffer[cPos - 1]:= #0;

    { save it! }
        processString:= StrPas(readBuffer);
        LineGot:= true;

    { if it's the last line of a reply, looks like xxx<space>blahblah... }
        if processString[4] = #32 then begin
        { save reply code }
           try
              if ServerReply.Count = 0 then { if it's a single line reply... }
                 LastReplyCode:= StrToInt(Copy(processString, 1, 3))
              else { multi-lined... the first number is the most important }
                 LastReplyCode:= StrToInt(Copy(ServerReply[0], 1, 3));
           except
              on EConvertError do LastReplyCode:= $FFFF;
           end;
        { now we've got the reply, sign it... }
           if LastReplyCode <> $FFFF then begin
              ReplyGot:= true;
              ReceivingReply:= false;
              if not ExpectedReply and Assigned(OnUnexpectedReply) then
              begin
                 ServerReply.Add(processString);
                 OnUnexpectedReply;
              end;
              ExpectedReply:= false;
              ReadBufferTop:= 0;
           end;
        end;

        ServerReply.Add(processString);
        readBuffer[cPos]:= saveChar;

    { move data down in the buffer 'window' }
        move(readBuffer[cPos + 1], readBuffer, sizeOf(readBuffer) - cPos - 1);
        receivedBytes := receivedBytes - cPos;
   end;
end;

{------------------------------------------------------------------------------}

function TControlConnection.GetServerReply(TimeOut: byte): TStringList;
var savedErrorCode: word;
begin
{ if we've got reply, just return with reference of private ServerReply }
     if ReplyGot then begin
        Result:= ServerReply;
        ReplyGot:= false;
        exit;
     end;

{ not connected and no reply got? shit... }
     if not Connected then
        raise ECNotConnected.Create('Can''t receive server reply - control ' +
                                     'socket closed.');

{ start timer }
     StartTimer(TimeOut);
     while not ReplyGot and not HasTimedOut(TimeOut) and (wsLastError= 0) do
     begin
           Application.ProcessMessages;
           if Application.Terminated then Exit;
           if Terminating then Exit;
     end;
{ stop timer }
     StopTimer;

{ if we've reached the TimeOut limit, then throw exception ECReplyTimedOut }
     if TimeSpent >= TimeOut then
        raise ECReplyTimedOut.Create('Server reply timed out.');

{ throw ECSocketException on any socket error occured }
     if wsLastError > 0 then begin
        savedErrorCode:= wsLastError;
        wsLastError:= 0;
        raise ECSocketException.CreateFmt('Socket abstract error: %d.',
              [savedErrorCode]);
     end;

{ if there was no error... happiness forever: Result set to ServerReply }
     Result:= ServerReply;
     ReplyGot:= false;
end;

{------------------------------------------------------------------------------}

function TControlConnection.GetServerReplyCode: word;
begin
{ Last reply code stored in private variable LastReplyCode }
     Result:= LastReplyCode;
end;

{------------------------------------------------------------------------------}

function TControlConnection.GetServerReplyType: byte;
begin
{ Determine reply type by the 3rd digit from right }
     if (LastReplyCode div 100) in [1..5] then
        Result:= LastReplyCode div 100
     else
        Result:= srUNKNOWN_REPLY_TYPE;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.SetExpectReply(ExpectReply: boolean);
begin
     ExpectedReply:= ExpectReply;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
{------ constructor/destructor methods  ---------------------------------------}
{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}
constructor TControlConnection.Create(xRemoteHost, xRemotePort: string;
                                      AOwner: TComponent);
begin
     inherited Create;
     RemoteHost:= xRemoteHost;
     RemotePort:= xRemotePort;

{ create control socket set up remote host data }
     ControlSocket      := TWSocket.Create(AOwner);
     ControlSocket.Addr := RemoteHost;
     ControlSocket.Port := RemotePort;

{ create timeout timer }
     TimeOutTimer:= TTimer.Create(AOwner);
     TimeOutTimer.Enabled:= false;
     TimeOutTimer.OnTimer:= TimeOutTimerTick;

{ set socket event handlers }
     ControlSocket.OnSessionConnected := ControlSessionConnected;
     ControlSocket.OnDataAvailable    := ControlDataAvailable;
     ControlSocket.OnSessionClosed    := ControlSessionClosed;

{ set internal data }
     Connected       := false; { use ConnectToServer to connect }
     wsLastError     := 0;     { 0 means OK, no socket - no error :) }
     LastReplyCode   := $ffff; { invalid reply code set }
     ReceivingReply  := false; { do what I say :) }
     ReplyGot        := false; { we don't have a reply first... we hope :) }
     ExpectedReply   := false; { yeah. A closed connection doesn't get replies }
     ReadBufferTop   := 0;     { buffer is empty }
     Terminating     := false; { we don't want to terminate first... }

{ external 'event handlers' }
     OnUnexpectedReply := nil;  { don't call not defined reference }
     OnControlClosed   := nil;  { yeah... i like nils}

{ create TStringlist contains the server's reply }
     ServerReply:= TStringList.Create;

end;

{------------------------------------------------------------------------------}

destructor TControlConnection.Destroy;
begin
{ Shut down I/O }
     ShutDownIO;
{ ... free other objects }
     if Assigned(ServerReply) then ServerReply.Free;
     inherited Destroy;
end;

{------------------------------------------------------------------------------}

procedure TControlConnection.Free;
begin
     if Assigned(Self) then Destroy;
end;

end.

